/*
 * Simple program to test HPSS c wrapper functions.
 *
 * Command line options:
 *    -m name	Set mailbox name.
 *    -v name   On each connect, show CGI variable named.
 *    -f name   On each connect, show form symbosl.
 *    -d flags  Dump flags: 0 - message, 1 - content, 2 - CGI table, 3 - forms
 *    -t secs   Set accept timeout seconds.
 *    -x type   Set mailbox type
 *
 * Author:  David Jones
 * Date:    6-FEB-2000
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ssdef.h>		/* VMS system service return codes */
#include "hpss_share.h"
/*
 * Declare variables that are set by command line arguments.
 */
static char **var_list;		/* CGI variables to display. */
static char **form_list;	/* form symbols to display */
static int timeout_secs;
static int mailbox_type;
static int dump_options = 0;
static char *mailbox_name = "HPSS_SRV_DEMO";

static char *usage_text = "Usage: hpss_demo {options}\n\
  -m name     Set mailbox name.\n\
  -v name     On each connect, show CGI variable named.\n\
  -f name     On each connect, show form symbols.\n\
  -d flags    Dump data: 0-msg 1-content 2-CGI 3-form\n\
  -t secs     Set accept timeout seconds.\n\
  -x type     Set mailbox type.\n";

/*
 * Parse command line arguments.
 */
static int parse_options ( int argc, char **argv )
{
    int i, vcount, fcount;
    /*
     * initialize.
     */
    var_list = (char **) malloc ( argc * sizeof(char *) );
    form_list = (char **) malloc ( argc * sizeof(char *) );
    vcount = fcount = 0;
    mailbox_type = 0;
    timeout_secs = 0;
    dump_options = 0;
    /*
     * Scan command line arguments.
     */
    for ( i = 1; i < argc; i+=2 ) {
	if ( argv[i][0] != '-' ) {
	    fprintf ( stderr, usage_text );
	    return 0;
	} else if ( argv[i][1] && argv[i][2] ) {	/* more than 1 letter */
	    fprintf ( stderr, usage_text );
	    return 0;
	} else if ( (i+1) >= argc ) {		/* missing argument */
	    fprintf ( stderr, usage_text );
	    return 0;
	}
	switch ( argv[i][1] ) {
	    case 'm':
	    case 'M':
		mailbox_name = argv[i+1];
		break;
	     case 'v':
	    case 'V':
		var_list[vcount++] = argv[i+1];
		break;
	    case 'f':
	    case 'F':
		form_list[fcount++] = argv[i+1];
		break;
	    case 't':
	    case 'T':
		timeout_secs = atoi ( argv[i+1] );
		break;
	    case 'x':
	    case 'X':
		mailbox_type = atoi ( argv[i+1] );
		break;
	    case 'd':
	    case 'D':
		dump_options = atoi ( argv[i+1] );
		break;
	    default:
		fprintf ( stderr, usage_text );
		return 0;
	}
    }
    var_list[vcount] = (char *) 0;	/* terminate list */
    form_list[fcount] = (char *) 0;
    return 1;
}
/*
 * Parse list of names and lookup the env value for each name.
 */
static void display_env_list ( int *context, char *list )
{
    char name[256], value[256];
    int status, i, nlen, j, length;

    nlen = 0;
    for ( i = 0; ; i++ ) {
	name[nlen] = list[i];
	if ( (name[nlen] == ',') || (name[nlen] == '\0') ) {
	    /*
	     * Reached end of name.
	     */
	    name[nlen] = '\0';
	    nlen = 0;
	    status = hpss_getenv_c ( context, name, value, sizeof(value), 
			&length );
	    if ( (status&1) == 1 ) {
		/* printf ( "    %s = '%s' (%d)\n", name, value, length ); */
		status = hpss_printf_c ( context,
			"    %s = '%s' (%d)\n", name, value, length );
	    } else {
		/* printf ( "    %s not found, error code: %d\n", name, 
			status ); */
		status = hpss_printf_c ( context,
			"    %s not found, error code: %d\n", name, status );
	    }
	    if ( (status&1) == 0 ) break;
	} else {
	    nlen++;
	    if ( nlen >= sizeof(name) ) {
		/* printf ( "    Bad name in target list, aborting!\n" ); */
		hpss_printf_c ( context, 
			"    Bad name in target list, aborting!\n" );
		break;
	    }
	}
	if ( !list[i] ) break;		/* end of list reached */
    }
}
/**************************************************************************/
static void display_form_list ( int *context, char *list )
{
    char name[256], value[256];
    int status, i, nlen, j, length;

    nlen = 0;
    for ( i = 0; ; i++ ) {
	name[nlen] = list[i];
	if ( (name[nlen] == ',') || (name[nlen] == '\0') ) {
	    /*
	     * Reached end of name.
	     */
	    name[nlen] = '\0';
	    nlen = 0;
	    status = hpss_getform_c ( context, name, value, sizeof(value), 
			&length );
	    if ( (status&1) == 1 ) {
		/* printf ( "    %s = '%s' (%d)\n", name, value, length ); */
		status = hpss_printf_c ( context,
			"    %s = '%s' (%d)\n", name, value, length );
	    } else {
		/* printf ( "    %s not found, error code: %d\n", name, status); */
		status = hpss_printf_c ( context,
			"    %s not found, error code: %d\n", name, status );
	    }
	    if ( (status&1) == 0 ) break;
	} else {
	    nlen++;
	    if ( nlen >= sizeof(name) ) {
		/* printf ( "    Bad name in target list, aborting!\n" ); */
		hpss_printf_c ( context, 
			"    Bad name in target list, aborting!\n" );
		break;
	    }
	}
	if ( !list[i] ) break;		/* end of list reached */
    }
}
/**************************************************************************/
int main ( int argc, char **argv )
{
    int status, context, pid, i, length;
    char subfunc[32], value[256], symbol_list[1024];

    if ( 0 == parse_options ( argc, argv ) ) return 20;

    status = hpss_initialize_c ( mailbox_name, mailbox_type, &context );

    while ( status&1 ) {
	status = hpss_accept_c (&context, timeout_secs,  &pid, 
		subfunc, sizeof(subfunc));
	printf("Status of accept: %d, pid: %x, subfunc: %s\n", status, pid,
		subfunc );
	if ( (status&1) == 1 ) {
	    hpss_write_c ( &context, "Content-type: text/plain", 24, 1 );
	    hpss_write_c ( &context, "", 0, 1 );
	    /*
	     * Force ACCVIO if path_info is crash (test recovery).
	     */
	    value[0] = '\0';
	    status = hpss_getenv_c ( &context, "PATH_INFO", value,
		sizeof(value), &length );
	    if ( strcmp ( value, "/crash" ) == 0 ) {
		printf("Forcing fatal exception...\n");
		display_env_list ( &context, (char *) 0 );
	    }
	    /*
	     * Display CGI variables if any specified.
	     */
	    if ( var_list[0] ) {
		/* printf ( "CGI variables: \n" ); */
		status = hpss_printf_c ( &context, "CGI variables: \n" );

		for ( i = 0; var_list[i]; i++ ) {
		    if ( strcmp ( var_list[i], "*" ) == 0 ) {
			status = hpss_getenv_c ( &context, "%SYMBOL_LIST",
			    symbol_list, sizeof(symbol_list), &length );
			if ( (status&1) == 1 ) {
			    display_env_list ( &context, symbol_list );
			} else {
			    /* printf ( "    %s not returned, error: %d\n",
				var_list[i], status ); */
			    status = hpss_printf_c ( &context,
			    	"    %s not returned, error: %d\n",
				var_list[i], status );
			}
		    } else {
		        display_env_list ( &context, var_list[i] );
		    }
		}
		/* printf ( "\n" ); */
		hpss_write_c ( &context, "", 0, 1 );	/* blank line */
	    }
	    /*
	     * Display form data.
	     */
	    if ( form_list[0] ) {
		/* printf ( "Form symbols: \n" ); */
		status = hpss_printf_c ( &context, "Form symbols: \n" );

		for ( i = 0; form_list[i]; i++ ) {
		    if ( strcmp ( form_list[i], "*" ) == 0 ) {
			status = hpss_getform_c ( &context, "%SYMBOL_LIST",
			    symbol_list, sizeof(symbol_list), &length );
			if ( (status&1) == 1 ) {
			    display_form_list ( &context, symbol_list );
			} else {
			    /* printf ( "    %s not returned, error: %d\n",
				form_list[i], status ); */
			    status = hpss_printf_c ( &context,
			    	"    %s not returned, error: %d\n",
				form_list[i], status );
			}
		    } else {
		        display_env_list ( &context, var_list[i] );
		    }
		}
		/* printf ( "\n" );*/
		hpss_write_c ( &context, "", 0, 1 ); /* blank line */
	    }
	    hpss_write_c ( &context, "Last line.", 10, 1 );
	    hpss_dump ( &context, &dump_options );
	    status = hpss_disconnect ( &context );
	    if ( (status&1) == 0 ) fprintf(stderr,"Error in disconnect: %d\n",
		status );
	    if ( strcmp ( value, "/crash2" ) == 0 ) {
		printf("Forcing fatal exception...\n");
		display_env_list ( &context, (char *) 0 );
	    }
	} else if ( status == SS$_TIMEOUT ) {
	    /*
	     * Change status so we don't exit loop
	     */
	    status = SS$_NORMAL;
	}
    }
    return status;
}
