/*\
 * Modified for VMS Mosaic by GEC on 12/12/99
 *
 * $Id: bmptoppm.c,v 1.10 1992/11/24 19:38:17 dws Exp dws $
 * 
 * bmptoppm.c - Converts from a Microsoft Windows or OS/2 .BMP file to a
 * PPM file.
 * 
 * The current implementation is probably not complete, but it works for
 * all the BMP files I have.  I welcome feedback.
 * 
 * Copyright (C) 1992 by David W. Sanderson.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.  This software is provided "as is"
 * without express or implied warranty.
 * 
\*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include "readbmp.h"

#ifndef DISABLE_TRACE
extern int srcTrace;
extern int reportBugs;
#endif

#define	MAXCOLORS   	256

typedef unsigned char pixval;

typedef struct {
    pixval r, g, b;
} pixel;

static char     er_read[] = "BMP: read error\n";
static char     er_internal[] = "BMP: %s: internal error!\n";

#define PPM_ASSIGN(p,red,grn,blu) do { (p).r = (red); (p).g = (grn); (p).b = (blu); } while ( 0 )

/*
 * Classes of BMP files
 */
#define C_WIN     1
#define C_OS2     2
#define C_OS2_NEW 3

static unsigned long
BMPleninfoheader(class)
	int             class;
{
	switch (class) {
	case C_WIN:
		return 40;
	case C_OS2:
		return 12;
	case C_OS2_NEW:
		return 64;
	default:
#ifndef DISABLE_TRACE
		if (reportBugs) {
			fprintf(stderr, er_internal, "BMPleninfoheader");
		}
#endif
		return 0;
	}
}


static unsigned long
BMPlenrgbtable(class, bitcount, colors)
	int             class;
	unsigned long   bitcount;
	int		colors;
{
	unsigned long   lenrgb;

	if (bitcount < 1) {
#ifndef DISABLE_TRACE
		if (reportBugs) {
			fprintf(stderr, er_internal, "BMPlenrgbtable");
		}
#endif
		return 0;
	}
	switch (class) {
	case C_WIN:
	case C_OS2_NEW:
		lenrgb = 4;
		break;
	case C_OS2:
		lenrgb = 3;
		break;
	default:
#ifndef DISABLE_TRACE
		if (reportBugs) {
			fprintf(stderr, er_internal, "BMPlenrgbtable");
		}
#endif
		return 0;
	}

	return colors ? (colors * lenrgb) : ((1 << bitcount) * lenrgb);
}

/*
 * Length, in bytes, of a line of the image
 * 
 * Evidently each row is padded on the right as needed to make it a
 * multiple of 4 bytes long.  This appears to be true of both
 * OS/2 and Windows BMP files.
 */
static unsigned long
BMPlenline(class, bitcount, x)
	int             class;
	unsigned long   bitcount;
	unsigned long   x;
{
	unsigned long   bitsperline;

	bitsperline = x * bitcount;

	/*
	 * If bitsperline is not a multiple of 32, then round
	 * bitsperline up to the next multiple of 32.
	 */
	if ((bitsperline % 32) != 0) {
		bitsperline += (32 - (bitsperline % 32));
	}

	if ((bitsperline % 32) != 0) {
#ifndef DISABLE_TRACE
		if (reportBugs) {
			fprintf(stderr, er_internal, "BMPlenline");
		}
#endif
		return 0;
	}

	/* Number of bytes per line == bitsperline/8 */
	return bitsperline >> 3;
}

/* Return the number of bytes used to store the image bits */
static unsigned long
BMPlenbits(class, bitcount, x, y)
	int             class;
	unsigned long   bitcount;
	unsigned long   x;
	unsigned long   y;
{
	return y * BMPlenline(class, bitcount, x);
}

/* Return the offset to the BMP image bits */
static unsigned long
BMPoffbits(class, bitcount, colors)
	int             class;
	unsigned long   bitcount;
	int		colors;
{
	return 14 + BMPleninfoheader(class) +
		BMPlenrgbtable(class, bitcount, colors);
}

/* Return the size of the BMP file in bytes */
static unsigned long
BMPlenfile(class, bitcount, x, y, colors)
	int             class;
	unsigned long   bitcount;
	unsigned long   x;
	unsigned long   y;
	int		colors;
{
	return BMPoffbits(class, bitcount, colors)
		+ BMPlenbits(class, bitcount, x, y);
}

/*\
 * $Id: bitio.c,v 1.5 1992/11/24 19:36:46 dws Exp dws $
 *
 * bitio.c - bitstream I/O
 *
 * Works for (sizeof(unsigned long)-1)*8 bits.
 *
 * Copyright (C) 1992 by David W. Sanderson.
\*/

struct bitstream
{
	FILE 		*f;		/* bytestream */
	unsigned long	bitbuf;		/* bit buffer */
	int		nbitbuf;	/* number of bits in 'bitbuf' */
	char		mode;
};

typedef struct bitstream        *BITSTREAM;

#define Mask(n)		((1<<(n))-1)

#define BitPut(b,ul,n)	((b)->bitbuf = (((b)->bitbuf<<(n))	\
					|((ul)&Mask(n))),	\
			(b)->nbitbuf += (n))

#define BitGet(b,n)	(((b)->bitbuf>>((b)->nbitbuf-=(n))) & Mask(n))

/*
 * pm_bitinit() - allocate and return a struct bitstream * for the
 * given FILE*.
 *
 * mode must be one of "r" or "w", according to whether you will be
 * reading from or writing to the struct bitstream *.
 *
 * Returns 0 on error.
 */

static struct bitstream *
pm_bitinit(f, mode)
	FILE           *f;
	char           *mode;
{
	struct bitstream *ans = (struct bitstream *)0;

	if (!f || !mode || !*mode)
		return ans;
	if (strcmp(mode, "r") && strcmp(mode, "w"))
		return ans;

	ans = (struct bitstream *)calloc(1, sizeof(struct bitstream));
	if (ans) {
		ans->f = f;
		ans->mode = *mode;
	}

	return ans;
}

/*
 * pm_bitfini() - deallocate the given struct bitstream *.
 *
 * You must call this after you are done with the struct bitstream *.
 * 
 * It may flush some bits left in the buffer.
 *
 * Returns the number of bytes written, -1 on error.
 */

static int
pm_bitfini(b)
	struct bitstream *b;
{
	int nbyte = 0;

	if (!b)
		return -1;

	/* Flush the output */
	if (b->mode == 'w') {
		/* Flush the bits */
		if (b->nbitbuf < 0 || b->nbitbuf >= 8) {
			/* pm_bitwrite() didn't work */
			return -1;
		}

		/*
		 * If we get to here, nbitbuf is 0..7
		 */
		if (b->nbitbuf)	{
			char	c;

			BitPut(b, 0, (long)8-(b->nbitbuf));
			c = (char) BitGet(b, (long)8);
			if (putc(c, b->f) == EOF) {
				return -1;
			}
			nbyte++;
		}
	}

	free(b);
	return nbyte;
}

/*
 * pm_bitread() - read the next nbits into *val from the given file.
 * 
 * The last pm_bitread() must be followed by a call to pm_bitfini().
 * 
 * Returns the number of bytes read, -1 on error.
 */

static int 
pm_bitread(b, nbits, val)
	struct bitstream *b;
	unsigned long   nbits;
	unsigned long  *val;
{
	int nbyte = 0;
	int c;

	if (!b)
		return -1;

	while (b->nbitbuf < nbits) {
		if ((c = getc(b->f)) == EOF) {
			return -1;
		}
		nbyte++;

		BitPut(b, c, (long) 8);
	}

	*val = BitGet(b, nbits);
	return nbyte;
}


static int
GetByte(fp)
	FILE *fp;
{
	int v;

	if ((v = fgetc(fp)) == EOF) {
#ifndef DISABLE_TRACE
		if (reportBugs) {
			fprintf(stderr, "BMP: EOF in GetByte\n");
		}
#endif
	}

	return v;
}

static int
readlittleshort( in, sP )
    FILE *in;
    short *sP;
{
    int c;

    if ((c = fgetc(in)) == EOF)
        return -1;
    *sP = c & 0xff;
    if ((c = fgetc(in)) == EOF)
        return -1;
    *sP |= (c & 0xff) << 8;
    return 0;
}

static short
GetShort(fp)
	FILE           *fp;
{
	short           v;

	if (readlittleshort(fp, &v) == -1) {
#ifndef DISABLE_TRACE
		if (reportBugs) {
			fprintf(stderr, er_read);
		}
#endif
	}

	return v;
}

static int
readlittlelong( in, lP )
    FILE *in;
    long *lP;
{
    int c;

    if ((c = fgetc(in)) == EOF)
        return -1;
    *lP = c & 0xff;
    if ((c = fgetc(in)) == EOF)
        return -1;
    *lP |= ( c & 0xff ) << 8;
    if ((c = fgetc(in)) == EOF)
        return -1;
    *lP |= (c & 0xff) << 16;
    if ((c = fgetc(in)) == EOF)
        return -1;
    *lP |= (c & 0xff) << 24;
    return 0;
}

static long
GetLong(fp)
	FILE   *fp;
{
	long   v;

	if (readlittlelong(fp, &v) == -1) {
#ifndef DISABLE_TRACE
		if (reportBugs) {
			fprintf(stderr, er_read);
		}
#endif
	}

	return v;
}

/*
 * readto - read as many bytes as necessary to position the
 * file at the desired offset.
 */

static void
readto(fp, ppos, dst)
	FILE           *fp;
	unsigned long  *ppos;	/* Pointer to number of bytes read from fp */
	unsigned long   dst;
{
	unsigned long   pos;

	if (!fp || !ppos)
		return;

	pos = *ppos;

	if (pos > dst) {
#ifndef DISABLE_TRACE
		if (reportBugs) {
			fprintf(stderr, "BMP: internal error in readto()\n");
		}
#endif
	}

	for (; pos < dst; pos++) {
		if (fgetc(fp) == EOF) {
#ifndef DISABLE_TRACE
			if (reportBugs) {
				fprintf(stderr, er_read);
			}
#endif
		}
	}

	*ppos = pos;
}


/*
 * BMP reading routines
 */

static int
BMPreadfileheader(fp, ppos, poffBits)
	FILE           *fp;
	unsigned long  *ppos;	/* Number of bytes read from fp */
	unsigned long  *poffBits;
{
	unsigned long   cbSize;
	unsigned short  xHotSpot;
	unsigned short  yHotSpot;
	unsigned long   offBits;

	if (GetByte(fp) != 'B') {
#ifndef DISABLE_TRACE
		if (srcTrace || reportBugs) {
			fprintf(stderr, "not a BMP file\n");
		}
#endif
		return 0;
	}
	if (GetByte(fp) != 'M') {
#ifndef DISABLE_TRACE
		if (srcTrace || reportBugs) {
			fprintf(stderr, "not a BMP file\n");
		}
#endif
		return 0;
	}

	cbSize = GetLong(fp);
	xHotSpot = GetShort(fp);
	yHotSpot = GetShort(fp);
	offBits = GetLong(fp);

	*poffBits = offBits;

	*ppos += 14;

	return 1;
}

static int
BMPreadinfoheader(fp, ppos, pcx, pcy, pcBitCount, pclass, pcolors, pcomp)
	FILE           *fp;
	unsigned long  *ppos;	/* Number of bytes read from fp */
	unsigned long  *pcx;
	unsigned long  *pcy;
	unsigned short *pcBitCount;
	int            *pclass;
	int	       *pcolors;
	int	       *pcomp;
{
	unsigned long   cbFix;
	unsigned short  cPlanes;
	unsigned long   cx;
	unsigned long   cy;
	unsigned short  cBitCount;
	unsigned long	cCompress;
	int             class;
	int	        colors;

	cbFix = GetLong(fp);

	switch (cbFix) {
	case 12:
		class = C_OS2;

		cx = GetShort(fp);
		cy = GetShort(fp);
		cPlanes = GetShort(fp);
		cBitCount = GetShort(fp);
		colors = 0;

		break;
	case 40:
		class = C_WIN;

		cx = GetLong(fp);
		cy = GetLong(fp);
		cPlanes = GetShort(fp);
		cBitCount = GetShort(fp);
		cCompress = GetLong(fp);

		/*
		 * We've read 20 bytes so far, need to read 20 more
		 * for the required total of 40.
		 */

		GetLong(fp);
		GetLong(fp);
		GetLong(fp);
		colors = GetLong(fp);
		GetLong(fp);

		break;
	case 64:
		class = C_OS2_NEW;

		cx = GetLong(fp);
		cy = GetLong(fp);
		cPlanes = GetShort(fp);
		cBitCount = GetShort(fp);
		cCompress = GetLong(fp);

		/*
		 * We've read 20 bytes so far, need to read 44 more
		 * for the required total of 64.
		 */

		GetLong(fp);
		GetLong(fp);
		GetLong(fp);
		colors = GetLong(fp);
		GetLong(fp);
		GetLong(fp);
		GetLong(fp);
		GetLong(fp);
		GetLong(fp);
		GetLong(fp);
		GetLong(fp);

		break;

	default:
#ifndef DISABLE_TRACE
		if (srcTrace || reportBugs) {
			fprintf(stderr, "BMP: unknown cbFix: %d\n", cbFix);
		}
#endif
		return 0;
	}

	if (cPlanes != 1) {
#ifndef DISABLE_TRACE
		if (srcTrace || reportBugs) {
			fprintf(stderr,
				"BMP: don't know how to handle cPlanes = %d\n",
				cPlanes);
		}
#endif
		return 0;
	}

#ifndef DISABLE_TRACE
	if (srcTrace) switch (class) {
	case C_WIN:
		fprintf(stderr, "Windows BMP, %dx%dx%d\n", cx, cy, cBitCount);
		fprintf(stderr, "compression = %d colors = %d\n",
			cCompress, colors);
		break;
	case C_OS2:
		fprintf(stderr, "OS/2 BMP, %dx%dx%d\n", cx, cy, cBitCount);
		break;
	case C_OS2_NEW:
		fprintf(stderr, "New OS/2 BMP, %dx%dx%d\n", cx, cy, cBitCount);
		fprintf(stderr, "compression = %d colors = %d\n",
			cCompress, colors);
		break;
	}
#endif
	*pcx = cx;
	*pcy = cy;
	*pcBitCount = cBitCount;
	*pclass = class;
	*pcolors = colors;
	*pcomp = cCompress;

	*ppos += cbFix;

	return 1;
}

/*
 * Returns the number of bytes read, or -1 on error.
 */
static int
BMPreadrgbtable(fp, ppos, cBitCount, class, R, G, B, colors)
	FILE           *fp;
	unsigned long  *ppos;	/* Number of bytes read from fp */
	unsigned short  cBitCount;
	int             class;
	pixval         *R;
	pixval         *G;
	pixval         *B;
	int		colors;
{
	int             i;
	int		nbyte = 0;
	long            ncolors = colors ? colors : (1 << cBitCount);

	for (i = 0; i < ncolors; i++) {
		if (i < MAXCOLORS) {
			B[i] = GetByte(fp);
			G[i] = GetByte(fp);
			R[i] = GetByte(fp);
		} else {
			(void) GetByte(fp);
			(void) GetByte(fp);
			(void) GetByte(fp);
		}
		nbyte += 3;

		if (class != C_OS2) {
			GetByte(fp);
			nbyte++;
		}
	}

	*ppos += nbyte;
	return nbyte;
}

/*
 * Returns the number of bytes read, or -1 on error.
 */
static int
BMPreadrow(fp, ppos, row, cx, cBitCount, R, G, B)
	FILE           *fp;
	unsigned long  *ppos;	/* Number of bytes read from fp */
	unsigned char  *row;
	unsigned long   cx;
	unsigned short  cBitCount;
	pixval         *R;
	pixval         *G;
	pixval         *B;
{
	BITSTREAM       b;
	unsigned long   nbyte = 0;
	int             rc;
	unsigned long   x;

	if (cBitCount != 8) {
		unsigned long v;

		if ((b = pm_bitinit(fp, "r")) == (BITSTREAM) 0) {
			return -1;
		}

		for (x = 0; x < cx; x++) {
			if ((rc = pm_bitread(b, cBitCount, &v)) == -1) {
				return -1;
			}
			nbyte += rc;

			*row++ = (unsigned char) v;
		}

		if ((rc = pm_bitfini(b)) != 0) {
			return -1;
		}
	} else {
		int v;

		for (x = 0; x < cx; x++) {
			if ((v = fgetc(fp)) == EOF) {
				return -1;
			}
			nbyte++;
			*row++ = (unsigned char) v;
		}
	}
	/*
	 * Make sure we read a multiple of 4 bytes.
	 */
	while (nbyte % 4) {
		GetByte(fp);
		nbyte++;
	}

	*ppos += nbyte;
	return nbyte;
}

static unsigned char *BMPreadbits(fp, ppos, offBits, cx, cy, cBitCount, class, R, G, B)
	FILE           *fp;
	unsigned long  *ppos;	/* Number of bytes read from fp */
	unsigned long   offBits;
	unsigned long   cx;
	unsigned long   cy;
	unsigned short  cBitCount;
	int             class;
	pixval         *R;
	pixval         *G;
	pixval         *B;
{
	unsigned char  *image;
	unsigned char  *ptr;
	long            y;
	int		rc;

	readto(fp, ppos, offBits);

	image = (unsigned char *)malloc(cx * cy);

	if (!image)
		return NULL;

	/*
	 * The picture is stored bottom line first, top line last
	 */
	for (y = cy - 1; y >= 0; y--) {

		ptr = image + (y * cx);
		rc = BMPreadrow(fp, ppos, ptr, cx, cBitCount, R, G, B);

		if (rc == -1) {
#ifndef DISABLE_TRACE
			if (srcTrace || reportBugs) {
				fprintf(stderr,
					"BMP: couldn't read row %d\n", y);
			}
#endif
			/* Display what we got */
			break;
		}
		if (rc % 4) {
#ifndef DISABLE_TRACE
			if (srcTrace || reportBugs) {
				fprintf(stderr,
					"BMP: row had bad # of bytes: %d\n",rc);
			}
#endif
			return NULL;
		}
	}

	return image;
}


unsigned char *ReadBMP(FILE *ifp, int *width, int *height, XColor *colrs)
{
	int             rc;
	unsigned long	pos = 0;
	unsigned long   offBits;
	unsigned long   cx;
	unsigned long   cy;
	unsigned short  cBitCount;
	int             class;
	int		colors;
	int		comp;
	int		i, j, row;
	pixval          R[MAXCOLORS];	/* reds */
	pixval          G[MAXCOLORS];	/* greens */
	pixval          B[MAXCOLORS];	/* blues */
	unsigned char   *image;

	if (!BMPreadfileheader(ifp, &pos, &offBits))
		return NULL;

	if (!BMPreadinfoheader(ifp, &pos, &cx, &cy, &cBitCount, &class,
	    &colors, &comp))
		return NULL;

	if (cBitCount > 8) {
		fprintf(stderr, "BMP: cannot do %d bit\n", cBitCount);
		return NULL;
	}
	if (comp) {
		fprintf(stderr, "BMP: cannot do compressed\n");
		return NULL;
	}

	*width = cx;
	*height = cy;

#ifndef DISABLE_TRACE
	if (srcTrace || reportBugs) {
		if (offBits != BMPoffbits(class, cBitCount, colors)) {
			fprintf(stderr, "BMP: offBits is %d, expected %d\n",
				pos, BMPoffbits(class, cBitCount, colors));
		}
	}
#endif
	rc = BMPreadrgbtable(ifp, &pos, cBitCount, class, R, G, B, colors);

	for (i = 0; i < MAXCOLORS; i++) {
		colrs[i].red = R[i] * 256;
		colrs[i].green = G[i] * 256;
		colrs[i].blue = B[i] * 256;
		colrs[i].pixel = i;
		colrs[i].flags = DoRed|DoGreen|DoBlue;
	}

	image = BMPreadbits(ifp, &pos, offBits, cx, cy,
			     cBitCount, class, R, G, B);

#ifndef DISABLE_TRACE
	if (srcTrace || reportBugs) {
		if (pos != BMPlenfile(class, cBitCount, cx, cy, colors)) {
			fprintf(stderr,
				"BMP: read %d bytes, expected %d bytes\n",
				pos, BMPlenfile(class, cBitCount, cx, cy,
				 colors));
		}
	}
#endif

	return image;
}
